// Console.cpp : Defines the entry point for the console application.
//

#include "stdafx.h"

static const int W = 80, H = 50;

#define DRAW_PALETTE

#define SIZEOF(x) (sizeof(x)/sizeof(x[0]))

// xor-shift random number generator
namespace Random
{
	DWORD gSeed;
	void Seed(DWORD value)
	{
		gSeed = value;
	}
	DWORD Int(void)
	{
#if 0
		gSeed *= 1664525;
		gSeed += 1013904223;
#else
		gSeed ^= (gSeed << 13);
		gSeed ^= (gSeed >> 17);
		gSeed ^= (gSeed << 5);
#endif
		return gSeed;
	}
}

typedef BYTE StateRow[9];

int main(int argc, char **argv)
{
	int stateCount = 2;
	StateRow *newState = (StateRow *)calloc(256, sizeof(StateRow));

	if (argc > 1)
	{
		// start with "survival"
		int row = 1;
		for (char *p = argv[1]; *p != '\0'; ++p)
		{
			if (*p == 'c' || *p == 'C' || *p == ',' || row < 0)
			{
				// read state count
				stateCount = 0;
				for (++p; *p != '\0' && *p >= '0' && *p <= '9'; ++p)
					stateCount = stateCount * 10 + *p - '0';
				--p;

				if (stateCount > 2)
				{
					// convert death to "aging"
					for (BYTE c = 0; c < 9; ++c)
						newState[1][c] = newState[1][c] ? 1 : 2;
				
					// add generation transitions
					for (BYTE i = 2; i < stateCount - 1; ++i)
						for (BYTE c = 0; c < 9; ++c)
							newState[i][c] = BYTE(i + 1);
					for (BYTE c = 0; c < 9; ++c)
						newState[stateCount - 1][c] = 0;
				}
				else
				{
					// 2 is the minimum state count
					stateCount = 2;
				}

				continue;
			}
			else if (*p == 'b' || *p == 'B')
			{
				// select "birth" row
				row = 0;
			}
			else if (*p == 's' || *p == 'S')
			{
				// select "survival" row
				row = 1;
			}
			else if (*p == '/')
			{
				// go to the next part of S/B/C
				--row;
				if (row < 0)
					--p;
			}
			else if (*p == 'r' || *p == 'R')
			{
				// rule set state transitions
				row = 0;
				for (++p; *p != '\0' && *p >= '0' && *p <= '9'; ++p)
					row = row * 10 + *p - '0';
				if (*p == ':')
				{
					if (stateCount < row + 1)
					{
						stateCount = row + 1;
					}
					memset(&newState[row], 0, sizeof(StateRow));
					BYTE col = 0;
					BYTE val = 0;
					for (++p; *p != '\0'; ++p)
					{
						if (*p == ',')
							newState[row][col++] = BYTE(val), val = 0;
						else if (*p >= '0' && *p <= '9')
							val = val * 10 + *p - '0';
						else
							break;
					}
				}
			}
			else if (*p >= '0' && *p <= '8')
			{
				newState[row][*p - '0'] = 1;
			}
		}
	}
	else
	{
		// default to Conway's Life
		newState[0][3] = 1;
		newState[1][2] = 1;
		newState[1][3] = 1;
	}

	// set up the handles for reading/writing:
	HANDLE hOut = GetStdHandle(STD_OUTPUT_HANDLE);
	HANDLE hIn = GetStdHandle(STD_INPUT_HANDLE);

	// change the window title
	SetConsoleTitle(TEXT("CellularAutomata"));

	// change the console window size
	static const SMALL_RECT windowSize = { 0, 0, W - 1, H - 1 };
	SetConsoleWindowInfo(hOut, TRUE, &windowSize);

	// change the internal buffer size
	static const COORD bufferSize = { W, H };
	SetConsoleScreenBufferSize(hOut, bufferSize);

	// hide the cursor
	static const CONSOLE_CURSOR_INFO cursorInfo = { 100, FALSE };
	SetConsoleCursorInfo(hOut, &cursorInfo);

	// set up the cell buffer
	BYTE *cellMemory = (BYTE *)calloc(2, sizeof(BYTE) * H * W);
	BYTE *cellBuffer = &cellMemory[0], *cellBufferNext = &cellMemory[W * H];

	// cells
	CHAR_INFO *cellColor = (CHAR_INFO *)calloc(stateCount, sizeof(CHAR_INFO));
	cellColor[0].Attributes = 0;
	cellColor[1].Attributes = BACKGROUND_INTENSITY | BACKGROUND_BLUE | BACKGROUND_GREEN | BACKGROUND_RED;

	if (stateCount > 2)
	{
		static const BYTE colorRamp[] = {
			FOREGROUND_INTENSITY | FOREGROUND_BLUE | FOREGROUND_RED, 
			FOREGROUND_INTENSITY | FOREGROUND_BLUE,
//			FOREGROUND_INTENSITY | FOREGROUND_BLUE | FOREGROUND_GREEN,
			FOREGROUND_INTENSITY | FOREGROUND_GREEN,
			FOREGROUND_INTENSITY | FOREGROUND_GREEN | FOREGROUND_RED,
			FOREGROUND_INTENSITY | FOREGROUND_RED,
			FOREGROUND_BLUE | FOREGROUND_RED, 
			FOREGROUND_BLUE,
//			FOREGROUND_BLUE | FOREGROUND_GREEN,
			FOREGROUND_GREEN,
			FOREGROUND_GREEN | FOREGROUND_RED,
			FOREGROUND_RED,
		};
		static const BYTE charRamp[] = { 0x00, 0xB0, 0xB1, 0xB2 };

		int shiftFactor = 0;
		while ((stateCount - 2) >> shiftFactor > SIZEOF(colorRamp))
			++shiftFactor;
		int colorEnd = min((stateCount - 2) >> shiftFactor, SIZEOF(colorRamp));
		for (BYTE i = 0; i < stateCount - 2; ++i)
		{
			int entry = (i << 2) >> shiftFactor;
			cellColor[i+2].Char.AsciiChar = charRamp[entry & 3];
			cellColor[i+2].Attributes = colorRamp[entry >> 2] << 4;
			if ((entry >> 2) < colorEnd)
				cellColor[i+2].Attributes |= colorRamp[(entry >> 2) + 1];
		}

#ifdef DRAW_PALETTE
		// write the characters
		static const COORD writePos = { 0, 0 };
		COORD writeSize = { SHORT(stateCount), 1 };
		SMALL_RECT writeArea = { 0, 0, W - 1, 0 }; 
		WriteConsoleOutputA(hOut, cellColor, writeSize, writePos, &writeArea);
#endif
	}

	enum GameState
	{
		GAME_RUNNING,
		GAME_PAUSED,
		GAME_EDIT,
		GAME_RULE,
		GAME_QUIT
	};
	GameState gameState = GAME_EDIT;

	int delayCycles = 16;
	int delayCount = 0;
	BYTE drawState = 1;

    // last update time
    DWORD lastTime = timeGetTime();

	// seed the random generator
	Random::Seed(lastTime);

	while (gameState != GAME_QUIT)
	{
		bool stateUpdate = false;
		bool stateChanged = false;

		// if there are any pending input events
		DWORD numEvents = 0;
		if ((gameState != GAME_RUNNING) || (GetNumberOfConsoleInputEvents(hIn, &numEvents) && numEvents > 0))
		{
			INPUT_RECORD eventBuffer[16];	// = (INPUT_RECORD *)_alloca(sizeof(INPUT_RECORD) * numEvents);
			ReadConsoleInput(hIn, eventBuffer, sizeof(eventBuffer)/sizeof(eventBuffer[0]), &numEvents);
			for (DWORD i = 0; i < numEvents; ++i)
			{
				switch (eventBuffer[i].EventType)
				{
				case KEY_EVENT:
					if (eventBuffer[i].Event.KeyEvent.bKeyDown)
					{
						switch (eventBuffer[i].Event.KeyEvent.wVirtualKeyCode)
						{
						case VK_ESCAPE:
							// quit
							gameState = GAME_QUIT;
							break;

						case VK_UP:
							// speed up
							if (delayCycles > 0)
								--delayCycles;
							break;

						case VK_DOWN:
							// slow down
							++delayCycles;
							break;

						case VK_SPACE:
							// single-step
							if (gameState == GAME_RUNNING)
							{
								gameState = GAME_EDIT;
							}
							stateUpdate = true;
							break;

						case VK_RETURN:
							// toggle between run and edit
							if (gameState == GAME_RUNNING)
							{
								gameState = GAME_EDIT;
							}
							else if (gameState == GAME_EDIT)
							{
								gameState = GAME_RUNNING;
							}
							break;

						case VK_INSERT:
							// fill all cells
							for (int y = 0; y < H; ++y)
							{
								for (int x = 0; x < W; ++x)
								{
									cellBuffer[y * W + x] = drawState;
								}
							}
							stateChanged = true;
							break;

						case VK_DELETE:
							// clear all cells
							for (int y = 0; y < H; ++y)
							{
								for (int x = 0; x < W; ++x)
								{
									cellBuffer[y * W + x] = 0;
								}
							}
							stateChanged = true;
							break;

						case '1':
						case '2':
						case '3':
						case '4':
						case '5':
						case '6':
						case '7':
						case '8':
						case '9':
							// select draw state
							drawState = BYTE(eventBuffer[i].Event.KeyEvent.wVirtualKeyCode - '0');
							break;

						case '0':
							drawState = 10;
							break;

						case VK_F1:
						case VK_F2:
						case VK_F3:
						case VK_F4:
						case VK_F5:
						case VK_F6:
						case VK_F7:
						case VK_F8:
						case VK_F9:
						case VK_F10:
							// random fill pattern
							for (int y = 0; y < H; ++y)
							{
								for (int x = 0; x < W; ++x)
								{
									cellBuffer[y * W + x] = 
										((Random::Int() & (1023 >> (eventBuffer[i].Event.KeyEvent.wVirtualKeyCode - VK_F1))) == 0)
										^ ((eventBuffer[i].Event.KeyEvent.dwControlKeyState & SHIFT_PRESSED) != 0);
								}
							}
							stateChanged = true;
							break;
						}
					}
					break;

				case MOUSE_EVENT:
					// mouse click or move
					if (eventBuffer[i].Event.MouseEvent.dwEventFlags == 0 || 
						eventBuffer[i].Event.MouseEvent.dwEventFlags == MOUSE_MOVED)
					{
						// get button states and position
						COORD mousePos = eventBuffer[i].Event.MouseEvent.dwMousePosition;

						// if inside the window...
						if (mousePos.X >= 0 && mousePos.X < W &&
							mousePos.Y >= 0 && mousePos.Y < H)
						{
							// left click
							if (eventBuffer[i].Event.MouseEvent.dwButtonState & FROM_LEFT_1ST_BUTTON_PRESSED)
							{
								// draw a cell in the selected state
								cellBuffer[mousePos.Y * W + mousePos.X] = drawState;
								stateChanged = true;
							}

							// right click
							if (eventBuffer[i].Event.MouseEvent.dwButtonState & RIGHTMOST_BUTTON_PRESSED)
							{
								// clear a cell
								cellBuffer[mousePos.Y * W + mousePos.X] = 0;
								stateChanged = true;
							}
						}
					}
					break;

				case FOCUS_EVENT:
					// pause on losing focus
					if (eventBuffer[i].Event.FocusEvent.bSetFocus)
					{
						if (gameState == GAME_PAUSED)
							gameState = GAME_RUNNING;
					}
					else
					{
						if (gameState == GAME_RUNNING)
							gameState = GAME_PAUSED;
					}
					break;
				}
			}
		}

		if (gameState == GAME_RUNNING)
		{
			if (--delayCount < 0)
			{
				stateUpdate = true;
				delayCount = delayCycles;
			}
		}

		if (stateUpdate)
		{
			stateChanged = true;

			// update state
			for (int y = 0; y < W * H; y += W)
			{
				int ym = y>0 ? y-W : (H-1)*W;
				int yp = y<(H-1)*W ? y+W : 0;
				for (int x = 0; x < W; ++x)
				{
					int xm = x>0 ? x-1 : W-1;
					int xp = x<W-1 ? x+1 : 0;
					BYTE count =
						(cellBuffer[ym+xm] == 1) +
						(cellBuffer[ym+x] == 1) +
						(cellBuffer[ym+xp] == 1) +
						(cellBuffer[y+xm] == 1) +
						(cellBuffer[y+xp] == 1) +
						(cellBuffer[yp+xm] == 1) +
						(cellBuffer[yp+x] == 1) +
						(cellBuffer[yp+xp] == 1);
					cellBufferNext[y + x] = newState[cellBuffer[y + x]][count];
				}
			}

			// swap buffers
			BYTE *cellBufferTmp = cellBuffer;
			cellBuffer = cellBufferNext;
			cellBufferNext = cellBufferTmp;
		}

		if (stateChanged)
		{
			// set up the character buffer
			CHAR_INFO consoleBuffer[W*H];
			for (int i = 0; i < W * H; ++i)
			{
				consoleBuffer[i] = cellColor[cellBuffer[i]];
			}

			// write the characters
			static const COORD writePos = { 0, 0 };
			static const COORD writeSize = { W, H };
			SMALL_RECT writeArea = { 0, 0, W - 1, H - 1 }; 
			WriteConsoleOutputA(hOut, consoleBuffer, writeSize, writePos, &writeArea);
		}

		// try to run at 60 FPS
		int delayTime = 17 - timeGetTime() + lastTime;
		if (delayTime > 0)
			Sleep(delayTime);
		lastTime = timeGetTime();
	}

	free(cellColor);
	free(cellMemory);
	free(newState);

	return 0;
}
