// ConsoleLua.cpp : Defines the entry point for the console application.
//

#include "stdafx.h"

extern "C"
{
#include "Lua.h"
#include "lualib.h"
#include "lauxlib.h"
}

// performance counter
static LARGE_INTEGER perfFrequency;
static LARGE_INTEGER perfStart;

// standard out and in console handles
static HANDLE hOut, hIn;

// current button state
static WORD mouseButtons;

enum GameState
{
	GAME_RUNNING,
	GAME_PAUSED,
	GAME_QUIT
};
static GameState gameState = GAME_RUNNING;

// fnv-1a hash
inline unsigned int Hash(const unsigned char byte, unsigned int hash = 2166136261u)
{
	hash ^= byte;
	hash *= 16777619u;
	return hash;
}
inline unsigned int Hash(const char *string, unsigned int hash = 2166136261u)
{
	if (string == 0)
		return 0;
	for (const char *s = string; *s != 0; ++s)
		hash = Hash(unsigned char(tolower(*s)), hash);
	return hash;
}

// clamp
template<typename T> inline T Clamp(const T x, const T min, const T max)
{
	if (x < min)
		return min;
	if (x > max)
		return max;
	return x;
}

#if !defined LUA_VERSION_NUM || LUA_VERSION_NUM==501
/*
** Adapted from Lua 5.2.0
*/
#define lua_pushglobaltable(L) lua_pushvalue(L, LUA_GLOBALSINDEX)

extern "C" static void luaL_setfuncs(lua_State *L, const luaL_Reg *l, int nup) {
	luaL_checkstack(L, nup, "too many upvalues");
	for (; l->name != NULL; l++) {  /* fill the table with given functions */
		int i;
		for (i = 0; i < nup; i++)  /* copy upvalues to the top */
			lua_pushvalue(L, -nup);
		lua_pushstring(L, l->name);
		lua_pushcclosure(L, l->func, nup);  /* closure with those upvalues */
		lua_settable(L, -(nup + 3));
	}
	lua_pop(L, nup);  /* remove upvalues */
}

extern "C" void *luaL_testudata(lua_State *L, int ud, const char *tname) {
	void *p = lua_touserdata(L, ud);
	if (p != NULL) {  /* value is a userdata? */
		if (lua_getmetatable(L, ud)) {  /* does it have a metatable? */
			luaL_getmetatable(L, tname);  /* get correct metatable */
			if (!lua_rawequal(L, -1, -2))  /* not the same? */
				p = NULL;  /* value is a userdata with wrong metatable */
			lua_pop(L, 2);  /* remove both metatables */
			return p;
		}
	}
	return NULL;  /* value is not a userdata with a metatable */
}
#endif

// output to console standard out
static int ConsolePrint(char const *format, ...)
{
	va_list ap;
	va_start(ap, format);

	char buf[4096];
	int n = vsnprintf_s(buf, sizeof(buf), sizeof(buf), format, ap);

	DWORD written;
	WriteConsole(hOut, buf, strlen(buf), &written, NULL);

	va_end(ap);
	return n;
}

// output to debug console (if present)
static int DebugPrint(char const *format, ...)
{
	va_list ap;
	va_start(ap, format);

	char buf[4096];
	int n = vsnprintf_s(buf, sizeof(buf), sizeof(buf), format, ap);

	OutputDebugStringA(buf);

	DWORD written;
	WriteConsole(hOut, buf, strlen(buf), &written, NULL);

	va_end(ap);
	return n;
}

// debug handler
static int LuaDebugPrint(lua_State *L)
{
	int n = lua_gettop(L);  /* number of arguments */
	int i;
	lua_getglobal(L, "tostring");
	for (i = 1; i <= n; i++) {
		const char *s;
		size_t l;
		lua_pushvalue(L, -1);  /* function to be called */
		lua_pushvalue(L, i);   /* value to print */
		lua_call(L, 1, 1);
		s = lua_tolstring(L, -1, &l);  /* get result */
		if (s == NULL)
			return luaL_error(L,
			LUA_QL("tostring") " must return a string to " LUA_QL("print"));
		if (i>1) OutputDebugStringA("\t");
		OutputDebugStringA(s);
		lua_pop(L, 1);  /* pop result */
	}
	OutputDebugStringA("\n");
	return 0;
}

// output handler
// (color set by caller)
static int LuaOutput(lua_State *L)
{
	int n = lua_gettop(L);  /* number of arguments */
	int i;
	lua_getglobal(L, "tostring");
	for (i = 1; i <= n; i++) {
		const char *s;
		size_t l;
		lua_pushvalue(L, -1);  /* function to be called */
		lua_pushvalue(L, i);   /* value to print */
		lua_call(L, 1, 1);
		s = lua_tolstring(L, -1, &l);  /* get result */
		if (s == NULL)
			return luaL_error(L,
			LUA_QL("tostring") " must return a string to " LUA_QL("print"));
		if (i>1) ConsolePrint("\t");
		ConsolePrint("%s", s);
		lua_pop(L, 1);  /* pop result */
	}
	ConsolePrint("\n");
	return 0;
}

// print handler
// (use bright white to stand out against trace messages)
static int LuaPrint(lua_State *L)
{
	SetConsoleTextAttribute(hOut, FOREGROUND_RED | FOREGROUND_GREEN | FOREGROUND_BLUE | FOREGROUND_INTENSITY);
	return LuaOutput(L);
}

// warning handler
// (use bright yellow to stand out against trace messages)
static int LuaWarning(lua_State *L)
{
	SetConsoleTextAttribute(hOut, FOREGROUND_RED | FOREGROUND_GREEN | FOREGROUND_INTENSITY);
	LuaDebugPrint(L);
	return LuaOutput(L);
}

// panic handler
// (use magenta to indicate an unrecoverable error)
static int LuaPanic(lua_State *L)
{
	SetConsoleTextAttribute(hOut, FOREGROUND_RED | FOREGROUND_BLUE | FOREGROUND_INTENSITY);
	LuaDebugPrint(L);
	return LuaOutput(L);
}

// check status
// display error message if status is nonzero
// (use red to indicate a recoverable error)
static bool LuaCheckStatus(int status, lua_State *L, const char *format)
{
	if (status)
	{
		// error?
		const char *error = lua_tostring(L, -1);
		SetConsoleTextAttribute(hOut, FOREGROUND_RED | FOREGROUND_INTENSITY);
		DebugPrint(const_cast<char *>(format), error);
		ConsolePrint(const_cast<char *>(format), error);
		return false;
	}
	return true;
}

// get a handle from the Lua stack
static HANDLE GetHandle(lua_State *L, int n)
{
	return HANDLE(luaL_testudata(L, n, "HANDLE"));
}

// push a handle onto the Lua stack
static void PushHandle(lua_State *L, HANDLE h)
{
	if (h)
		lua_pushlightuserdata(L, (void *)(h));
	else
		lua_pushnil(L);
}

// handle to string
static int HANDLE_ToString(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	char buf[16];
	sprintf_s(buf, "%08X", h);
	lua_pushstring(L, buf);
	return 1;
}

// get character value
static CHAR GetChar(lua_State *L, int index)
{
	switch (lua_type(L, index))
	{
	case LUA_TNUMBER:
		return CHAR(lua_tointeger(L, index));
	case LUA_TSTRING:
		return lua_tostring(L, index)[0];
	default:
		return 0;
	}
}

// get attribute value
static WORD GetAttrib(lua_State *L, int index)
{
	switch (lua_type(L, index))
	{
	case LUA_TNUMBER:
		return WORD(lua_tointeger(L, index));
	case LUA_TSTRING:
		return WORD(lua_tostring(L, index)[0]);
	default:
		return 0;
	}
}

// get a CHAR_INFO from the Lua stack
// returns the number of values consumed
static int GetCharInfo(CHAR_INFO &info, lua_State *L, int index)
{
	const char *string;
	size_t length;
	switch (lua_type(L, index))
	{
	case LUA_TNUMBER:
		// get character and attribute from a pair of integers
		info.Char.AsciiChar = CHAR(luaL_checkinteger(L, index));
		info.Attributes = WORD(luaL_checkinteger(L, index + 1));
		return 2;
	case LUA_TSTRING:
		// get character and attribute from a string
		string = lua_tolstring(L, index, &length);
		if (length > 0)
		{
			info.Char.AsciiChar = string[0];
			if (length > 1)
			{
				info.Attributes = string[1];
			}
		}
		return 1;
	case LUA_TTABLE:
		// get character and attribute from array
		lua_rawgeti(L, index, 1);
		info.Char.AsciiChar = CHAR(luaL_optinteger(L, -1, info.Char.AsciiChar));
		lua_pop(L, 1);
		lua_rawgeti(L, index, 2);
		info.Attributes = WORD(luaL_optinteger(L, -1, info.Attributes));
		lua_pop(L, 1);
		// get character and attribute from table
		lua_getfield(L, index, "Char");
		info.Char.AsciiChar = CHAR(luaL_optinteger(L, -1, info.Char.AsciiChar));
		lua_pop(L, 1);
		lua_getfield(L, index, "Attrib");
		info.Attributes = WORD(luaL_optinteger(L, -1, info.Attributes));
		lua_pop(L, 1);
		return 1;
	default:
		// skip argument
		return 1;
	}
}

// get a COORD from the Lua stack
// returns the number of values consumed
static int GetCoord(COORD &coord, lua_State *L, int index)
{
	switch (lua_type(L, index))
	{
	case LUA_TNUMBER:
		// get coord from a pair of integers
		coord.X = SHORT(luaL_checkinteger(L, index));
		coord.Y = SHORT(luaL_checkinteger(L, index + 1));
		return 2;
	case LUA_TTABLE:
		// get coord from array
		lua_rawgeti(L, index, 1);
		coord.X = SHORT(luaL_optinteger(L, -1, coord.X));
		lua_pop(L, 1);
		lua_rawgeti(L, index, 2);
		coord.Y = SHORT(luaL_optinteger(L, -1, coord.Y));
		lua_pop(L, 1);
		// get coord from table
		lua_getfield(L, index, "X");
		coord.X = SHORT(luaL_optinteger(L, -1, coord.X));
		lua_pop(L, 1);
		lua_getfield(L, index, "Y");
		coord.Y = SHORT(luaL_optinteger(L, -1, coord.Y));
		lua_pop(L, 1);
		return 1;
	default:
		// skip argument
		return 1;
	}
}

// push a COORD onto the Lua stack
static void PushCoord(lua_State *L, COORD coord)
{
	lua_createtable(L, 0, 2);
	lua_pushinteger(L, coord.X);
	lua_setfield(L, -2, "X");
	lua_pushinteger(L, coord.Y);
	lua_setfield(L, -2, "Y");
}

// get SMALL_RECT from the Lua stack
// returns the number of values consumed
static int GetSmallRect(SMALL_RECT &rect, lua_State *L, int index)
{
	switch (lua_type(L, index))
	{
	case LUA_TNUMBER:
		// get rect from four integers
		rect.Left = SHORT(luaL_checkinteger(L, index));
		rect.Top = SHORT(luaL_checkinteger(L, index + 1));
		rect.Right = SHORT(luaL_checkinteger(L, index + 2));
		rect.Bottom = SHORT(luaL_checkinteger(L, index + 3));
		return 4;
	case LUA_TTABLE:
		// get rect from array
		lua_rawgeti(L, index, 1);
		rect.Left = SHORT(luaL_optinteger(L, -1, rect.Left));
		lua_pop(L, 1);
		lua_rawgeti(L, index, 2);
		rect.Top = SHORT(luaL_optinteger(L, -1, rect.Top));
		lua_pop(L, 1);
		lua_rawgeti(L, index, 3);
		rect.Right = SHORT(luaL_optinteger(L, -1, rect.Right));
		lua_pop(L, 1);
		lua_rawgeti(L, index, 4);
		rect.Bottom = SHORT(luaL_optinteger(L, -1, rect.Bottom));
		lua_pop(L, 1);
		// get rect from table
		lua_getfield(L, index, "Left");
		rect.Left = SHORT(luaL_optinteger(L, -1, rect.Left));
		lua_pop(L, 1);
		lua_getfield(L, index, "Top");
		rect.Top = SHORT(luaL_optinteger(L, -1, rect.Top));
		lua_pop(L, 1);
		lua_getfield(L, index, "Right");
		rect.Right = SHORT(luaL_optinteger(L, -1, rect.Right));
		lua_pop(L, 1);
		lua_getfield(L, index, "Bottom");
		rect.Bottom = SHORT(luaL_optinteger(L, -1, rect.Bottom));
		lua_pop(L, 1);
		return 1;
	default:
		// skip argument
		return 1;
	}
}

// push a SMALL_RECT onto the Lua stack
static void PushSmallRect(lua_State *L, const SMALL_RECT &rect)
{
	lua_createtable(L, 0, 4);
	lua_pushinteger(L, rect.Left);
	lua_setfield(L, -2, "Left");
	lua_pushinteger(L, rect.Top);
	lua_setfield(L, -2, "Top");
	lua_pushinteger(L, rect.Right);
	lua_setfield(L, -2, "Right");
	lua_pushinteger(L, rect.Bottom);
	lua_setfield(L, -2, "Bottom");
}

// structure representing a 2D array of CHAR_INFO
struct CHAR_INFO_Buffer
{
	COORD size;
	CHAR_INFO buffer[0];
};

// create a CHAR_INFO_Buffer of the specified size
// the buffer contents are uninitialized
static CHAR_INFO_Buffer *NewCharInfoBuffer(lua_State *L, COORD size)
{
	CHAR_INFO_Buffer *buffer = static_cast<CHAR_INFO_Buffer *>(lua_newuserdata(L, sizeof(CHAR_INFO_Buffer) + size.X * size.Y * sizeof(CHAR_INFO)));
	luaL_getmetatable(L, "CHAR_INFO_Buffer");
	lua_setmetatable(L, -2);
	buffer->size = size;
	return buffer;
}

// create a CHAR_INFO_Buffer on the Lua stack
static int NewCharInfoBuffer(lua_State *L)
{
	int index = 1;
	COORD size = { 0, 0 };
	index += GetCoord(size, L, index);
	CHAR_INFO info = { 0, 0 };
	index += GetCharInfo(info, L, index);
	CHAR_INFO_Buffer *c = NewCharInfoBuffer(L, size);
	for (int y = 0; y < size.Y; ++y)
		for (int x = 0; x < size.X; ++x)
			c->buffer[y * size.X + x] = info;
	return 1;
}

// get a CHAR_INFO_Buffer from the Lua stack
static CHAR_INFO_Buffer *GetCharInfoBuffer(lua_State *L, int n)
{
	return static_cast<CHAR_INFO_Buffer *>(luaL_testudata(L, n, "CHAR_INFO_Buffer"));
}

// get an element from a CHAR_INFO_Buffer
// receives (userdata, x, y)
static int CHAR_INFO_Buffer_Get(lua_State *L)
{
	CHAR_INFO_Buffer *c = GetCharInfoBuffer(L, 1);
	if (!c)
		return 0;
	int index = 2;
	COORD pos = { 0, 0 };
	index += GetCoord(pos, L, index);
	const COORD &size = c->size;
	if (pos.X < 0 || pos.X >= size.X || pos.Y < 0 || pos.Y >= size.Y)
		return 0;
	CHAR_INFO &info = c->buffer[pos.Y * size.X + pos.X];
	const char value[2] = { info.Char.AsciiChar, char(info.Attributes) };
	lua_pushlstring(L, value, 2);
	return 1;
}
// set an element in a CHAR_INFO_Buffer
// receives (userdata, x, y, value)
static int CHAR_INFO_Buffer_Set(lua_State *L)
{
	CHAR_INFO_Buffer *c = GetCharInfoBuffer(L, 1);
	if (!c)
		return 0;
	int index = 2;
	COORD pos = { 0, 0 };
	index += GetCoord(pos, L, index);
	const COORD &size = c->size;
	if (pos.X < 0 || pos.X >= size.X || pos.Y < 0 || pos.Y >= size.Y)
		return 0;
	CHAR_INFO &info = c->buffer[pos.Y * size.X + pos.X];
	GetCharInfo(info, L, index);
	return 0;
}

// quit
static int Quit(lua_State *L)
{
	(L);
	gameState = GAME_QUIT;
	return 0;
}

// pause/resume
static int Pause(lua_State *L)
{
	gameState = lua_toboolean(L, 1) ? GAME_PAUSED : GAME_RUNNING;
	return 0;
}

// sleep for the specified duration
static int Sleep(lua_State *L)
{
	int duration = luaL_checkinteger(L, 1);
	if (duration >= 0)
		Sleep(duration);
	return 0;
}

// get elapsed milliseconds since start
static int GetMillis(lua_State *L)
{
	LARGE_INTEGER perfNow;
	QueryPerformanceCounter(&perfNow);
	lua_pushinteger(L, int((perfNow.QuadPart - perfStart.QuadPart) * 1000 / perfFrequency.QuadPart));
	return 1;
}

// return the last error message, if any
static int GetLastError(lua_State *L)
{
	DWORD error = GetLastError();
	if (error == S_OK)
		return 0;
	char buffer[1024] = { 0 };
	FormatMessageA(FORMAT_MESSAGE_FROM_SYSTEM, NULL, error, 0, buffer, sizeof(buffer), NULL);
	lua_pushstring(L, buffer);
	return 1;
}

// close a handle
static int CloseHandle(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	lua_pushboolean(L, CloseHandle(h));
	return 1;
}

///////////
// wincon.h

// is there really a need for this?
//static int WriteConsoleKeyEvent(lua_State *L)

static int ReadConsoleOutputA(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	CHAR_INFO_Buffer *c = GetCharInfoBuffer(L, 2);
	if (!c)
		return 0;
	int index = 3;
	COORD buffercoord = { 0, 0 };
	index += GetCoord(buffercoord, L, index);
	SMALL_RECT region = { 0, 0, 0, 0 };
	index += GetSmallRect(region, L, index);
	if (!ReadConsoleOutputA(h, c->buffer, c->size, buffercoord, &region))
		return 0;
	lua_pushnumber(L, region.Left);
	lua_pushnumber(L, region.Top);
	lua_pushnumber(L, region.Right);
	lua_pushnumber(L, region.Bottom);
	return 4;
}

static int WriteConsoleOutputA(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	const CHAR_INFO_Buffer *c = GetCharInfoBuffer(L, 2);
	if (!c)
		return 0;
	int index = 3;
	COORD buffercoord = { 0, 0 };
	index += GetCoord(buffercoord, L, index);
	SMALL_RECT region = { 0, 0, 0, 0 };
	index += GetSmallRect(region, L, index);
	if (!WriteConsoleOutputA(h, c->buffer, c->size, buffercoord, &region))
		return 0;
	lua_pushnumber(L, region.Left);
	lua_pushnumber(L, region.Top);
	lua_pushnumber(L, region.Right);
	lua_pushnumber(L, region.Bottom);
	return 4;
}

static int ReadConsoleOutputCharacterA(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	size_t length = luaL_checkinteger(L, 2);
	char *text = static_cast<char *>(alloca(length));
	COORD pos = { 0, 0 };
	GetCoord(pos, L, 3);
	DWORD read;
	if (!ReadConsoleOutputCharacterA(h, text, length, pos, &read))
		return 0;
	lua_pushlstring(L, text, read);
	return 1;
}

static int ReadConsoleOutputAttribute(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	size_t length = luaL_checkinteger(L, 2);
	WORD *attrib = static_cast<WORD *>(alloca(sizeof(WORD)*length));
	COORD pos = { 0, 0 };
	GetCoord(pos, L, 3);
	DWORD read;
	if (!ReadConsoleOutputAttribute(h, attrib, length, pos, &read))
		return 0;
	char *text = static_cast<char *>(alloca(length));
	for (size_t i = 0; i < length; ++i)
		text[i] = char(attrib[i]);
	lua_pushlstring(L, text, read);
	return 1;
}

static int WriteConsoleOutputCharacterA(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	size_t length = 0;
	const char * text = luaL_checklstring(L, 2, &length);
	COORD pos = { 0, 0 };
	GetCoord(pos, L, 3);
	DWORD written;
	if (!WriteConsoleOutputCharacterA(h, text, length, pos, &written))
		return 0;
	lua_pushinteger(L, written);
	return 1;
}

static int WriteConsoleOutputAttribute(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	size_t length = 0;
	const char * text = luaL_checklstring(L, 2, &length);
	WORD *attrib = static_cast<WORD *>(alloca(sizeof(WORD)*length));
	for (size_t i = 0; i < length; ++i)
		attrib[i] = WORD(text[i]);
	COORD pos = { 0, 0 };
	GetCoord(pos, L, 3);
	DWORD written;
	if (!WriteConsoleOutputAttribute(h, attrib, length, pos, &written))
		return 0;
	lua_pushinteger(L, written);
	return 1;
}

static int FillConsoleOutputCharacterA(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	char character = GetChar(L, 2);;
	int length = luaL_checkinteger(L, 3);
	COORD pos = { 0, 0 };
	GetCoord(pos, L, 4);
	DWORD written;
	if (!FillConsoleOutputCharacterA(h, character, length, pos, &written))
		return 0;
	lua_pushinteger(L, written);
	return 1;
}

static int FillConsoleOutputAttribute(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	WORD attrib = GetAttrib(L, 2);
	int length = luaL_checkinteger(L, 3);
	COORD pos = { 0, 0 };
	GetCoord(pos, L, 4);
	DWORD written;
	if (!FillConsoleOutputAttribute(h, attrib, length, pos, &written))
		return 0;
	lua_pushinteger(L, written);
	return 1;
}

static int GetConsoleScreenBufferInfo(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	CONSOLE_SCREEN_BUFFER_INFO info = { 0 };
	if (!GetConsoleScreenBufferInfo(h, &info))
		return 0;
	lua_createtable(L, 0, 5);
	PushCoord(L, info.dwSize);
	lua_setfield(L, -2, "Size");
	lua_createtable(L, 0, 2);
	PushCoord(L, info.dwCursorPosition);
	lua_setfield(L, -2, "CursorPosition");
	lua_pushinteger(L, info.wAttributes);
	lua_setfield(L, -2, "Attributes");
	PushSmallRect(L, info.srWindow);
	lua_setfield(L, -2, "Window");
	PushCoord(L, info.dwMaximumWindowSize);
	lua_setfield(L, -2, "MaximumWindowSize");
	return 1;
}

static int GetConsoleScreenBufferInfoEx(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	CONSOLE_SCREEN_BUFFER_INFOEX info = { 0 };
	info.cbSize = sizeof(CONSOLE_SCREEN_BUFFER_INFOEX);
	if (!GetConsoleScreenBufferInfoEx(h, &info))
		return 0;
	lua_createtable(L, 0, 8);
	PushCoord(L, info.dwSize);
	lua_setfield(L, -2, "Size");
	lua_createtable(L, 0, 2);
	PushCoord(L, info.dwCursorPosition);
	lua_setfield(L, -2, "CursorPosition");
	lua_pushinteger(L, info.wAttributes);
	lua_setfield(L, -2, "Attributes");
	PushSmallRect(L, info.srWindow);
	lua_setfield(L, -2, "Window");
	PushCoord(L, info.dwMaximumWindowSize);
	lua_setfield(L, -2, "MaximumWindowSize");
	lua_pushinteger(L, info.wPopupAttributes);
	lua_setfield(L, -2, "PopupAttributes");
	lua_pushboolean(L, info.bFullscreenSupported);
	lua_setfield(L, -2, "FullscreenSupported");
	lua_createtable(L, 16, 0);
	for (int i = 0; i < 16; ++i)
	{
		COLORREF color = info.ColorTable[i];
		lua_createtable(L, 3, 0);
		lua_pushinteger(L, LOBYTE(color));
		lua_rawseti(L, -2, 1);
		lua_pushinteger(L, LOBYTE(color >> 8));
		lua_rawseti(L, -2, 2);
		lua_pushinteger(L, LOBYTE(color >> 16));
		lua_rawseti(L, -2, 3);
		lua_rawseti(L, -2, i);
	}
	lua_setfield(L, -2, "ColorTable");
	return 1;
}

static int GetConsoleScreenBufferSize(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	CONSOLE_SCREEN_BUFFER_INFO info = { 0 };
	if (!GetConsoleScreenBufferInfo(h, &info))
		return 0;
	lua_pushinteger(L, info.dwSize.X);
	lua_pushinteger(L, info.dwSize.Y);
	return 2;
}

static int GetConsoleCursorPosition(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	CONSOLE_SCREEN_BUFFER_INFO info = { 0 };
	if (!GetConsoleScreenBufferInfo(h, &info))
		return 0;
	lua_pushinteger(L, info.dwCursorPosition.X);
	lua_pushinteger(L, info.dwCursorPosition.Y);
	return 2;
}

static int GetConsoleTextAttribute(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	CONSOLE_SCREEN_BUFFER_INFO info = { 0 };
	if (!GetConsoleScreenBufferInfo(h, &info))
		return 0;
	lua_pushinteger(L, info.wAttributes);
	return 1;
}

static int GetConsoleScreenBufferRect(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	CONSOLE_SCREEN_BUFFER_INFO info = { 0 };
	if (!GetConsoleScreenBufferInfo(h, &info))
		return 0;
	lua_pushinteger(L, info.srWindow.Left);
	lua_pushinteger(L, info.srWindow.Top);
	lua_pushinteger(L, info.srWindow.Right);
	lua_pushinteger(L, info.srWindow.Bottom);
	return 4;
}

static int GetConsolePopupAttributes(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	CONSOLE_SCREEN_BUFFER_INFOEX info = { 0 };
	info.cbSize = sizeof(CONSOLE_SCREEN_BUFFER_INFOEX);
	if (!GetConsoleScreenBufferInfoEx(h, &info))
		return 0;
	lua_pushinteger(L, info.wPopupAttributes);
	return 1;
}

static int GetConsoleFullscreenSupported(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	CONSOLE_SCREEN_BUFFER_INFOEX info = { 0 };
	info.cbSize = sizeof(CONSOLE_SCREEN_BUFFER_INFOEX);
	if (!GetConsoleScreenBufferInfoEx(h, &info))
		return 0;
	lua_pushboolean(L, info.bFullscreenSupported);
	return 1;
}

static int GetConsoleColorTable(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	CONSOLE_SCREEN_BUFFER_INFOEX info = { 0 };
	info.cbSize = sizeof(CONSOLE_SCREEN_BUFFER_INFOEX);
	if (!GetConsoleScreenBufferInfoEx(h, &info))
		return 0;
	if (lua_isnumber(L, 1))
	{
		int i = luaL_checkinteger(L, 1);
		if (i >= 0 && i < 16)
		{
			COLORREF color = info.ColorTable[i];
			lua_pushinteger(L, LOBYTE(color));
			lua_pushinteger(L, LOBYTE(color >> 8));
			lua_pushinteger(L, LOBYTE(color >> 16));
			return 3;
		}
		return 0;
	}
	else
	{
		lua_createtable(L, 16, 0);
		for (int i = 0; i < 16; ++i)
		{
			COLORREF color = info.ColorTable[i];
			lua_createtable(L, 3, 0);
			lua_pushinteger(L, LOBYTE(color));
			lua_rawseti(L, -2, 1);
			lua_pushinteger(L, LOBYTE(color >> 8));
			lua_rawseti(L, -2, 2);
			lua_pushinteger(L, LOBYTE(color >> 16));
			lua_rawseti(L, -2, 3);
			lua_rawseti(L, -2, i);
		}
		return 1;
	}
}

static DWORD GetColorValue(lua_State *L, int index, DWORD default)
{
	int r = LOBYTE(default), g = LOBYTE(default >> 8), b = LOBYTE(default >> 16);
	if (index < 0)
	{
		index += lua_gettop(L) + 1;
	}
	if (lua_isstring(L, index))
	{
		// get color from hexadecimal #RRGGBB
		const char *value = lua_tostring(L, 2);
		sscanf_s(value, "#%02x%02x%02x", &r, &g, &b);
	}
	else if (lua_istable(L, index))
	{
		// get color from array: { <red>, <green>, <blue> }
		lua_rawgeti(L, index, 1);
		if (lua_isnumber(L, -1))
			r = lua_tointeger(L, -1);
		lua_pop(L, 1);
		lua_rawgeti(L, index, 2);
		if (lua_isnumber(L, -1))
			g = luaL_checkinteger(L, -1);
		lua_pop(L, 1);
		lua_rawgeti(L, index, 3);
		if (lua_isnumber(L, -1))
			b = luaL_checkinteger(L, -1);
		lua_pop(L, 1);

		// get color from table: { R=<red>, G=<green>, B=<blue> }
		lua_getfield(L, index, "R");
		if (lua_isnumber(L, -1))
			r = lua_tointeger(L, -1);
		lua_pop(L, 1);
		lua_getfield(L, index, "G");
		if (lua_isnumber(L, -1))
			g = luaL_checkinteger(L, -1);
		lua_pop(L, 1);
		lua_getfield(L, index, "B");
		if (lua_isnumber(L, -1))
			b = luaL_checkinteger(L, -1);
		lua_pop(L, 1);
	}
	else if (index < lua_gettop(L) - 2)
	{
		r = luaL_checkinteger(L, 2);
		g = luaL_checkinteger(L, 3);
		b = luaL_checkinteger(L, 4);
	}
	return
		(Clamp(r, 0, 255)) |
		(Clamp(g, 0, 255) << 8) |
		(Clamp(b, 0, 255) << 16);
}

static int SetConsoleScreenBufferInfoEx(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	CONSOLE_SCREEN_BUFFER_INFOEX info = { 0 };
	info.cbSize = sizeof(CONSOLE_SCREEN_BUFFER_INFOEX);
	if (!GetConsoleScreenBufferInfoEx(h, &info))
		return 0;
	lua_getfield(L, 1, "Size");
	GetCoord(info.dwSize, L, -1);
	lua_pop(L, 1);
	lua_getfield(L, 1, "CursorPosition");
	GetCoord(info.dwCursorPosition, L, -1);
	lua_pop(L, 1);
	lua_getfield(L, 1, "Attributes");
	info.wAttributes = WORD(luaL_optinteger(L, -1, info.wAttributes));
	lua_pop(L, 1);
	lua_getfield(L, 1, "Window");
	GetSmallRect(info.srWindow, L, -1);
	lua_pop(L, 1);
	lua_getfield(L, 1, "MaximumWindowSize");
	GetCoord(info.dwMaximumWindowSize, L, -1);
	lua_pop(L, 1);
	lua_getfield(L, 1, "PopupAttributes");
	info.wPopupAttributes = WORD(luaL_optinteger(L, -1, info.wPopupAttributes));
	lua_pop(L, 1);
	lua_getfield(L, 1, "ColorTable");
	if (lua_istable(L, 1))
	{
		// set multiple color values
		lua_pushnil(L);
		while (lua_next(L, 1))
		{
			// key @-2, value @-1
			if (lua_isnumber(L, -2))
			{
				int i = lua_tointeger(L, -2);
				if (i >= 0 && i < 16)
				{
					info.ColorTable[i] = GetColorValue(L, -1, info.ColorTable[i]);
				}
			}
			lua_pop(L, 1);
		}
	}
	lua_pop(L, 1);
	lua_pushboolean(L, SetConsoleScreenBufferInfoEx(h, &info));
	return 1;
}

static int SetConsoleColorTable(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	CONSOLE_SCREEN_BUFFER_INFOEX info = { 0 };
	info.cbSize = sizeof(CONSOLE_SCREEN_BUFFER_INFOEX);
	if (!GetConsoleScreenBufferInfoEx(h, &info))
		return 0;

	if (lua_istable(L, 1))
	{
		// set multiple color values
		lua_pushnil(L);
		while (lua_next(L, 1))
		{
			// key @-2, value @-1
			if (lua_isnumber(L, -2))
			{
				int i = lua_tointeger(L, -2);
				if (i >= 0 && i < 16)
				{
					info.ColorTable[i] = GetColorValue(L, -1, info.ColorTable[i]);
				}
			}
			lua_pop(L, 1);
		}
	}
	else
	{
		// set one color value
		int i = luaL_checkinteger(L, 1);
		if (i >= 0 && i < 16)
		{
			info.ColorTable[i] = GetColorValue(L, 2, info.ColorTable[i]);
		}
	}

	++info.srWindow.Right;
	++info.srWindow.Bottom;
	lua_pushboolean(L, SetConsoleScreenBufferInfoEx(h, &info));
	return 1;
}

static int GetLargestConsoleWindowSize(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	COORD size = GetLargestConsoleWindowSize(h);
	lua_pushinteger(L, size.X);
	lua_pushinteger(L, size.Y);
	return 2;
}

static int GetConsoleCursorInfo(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	CONSOLE_CURSOR_INFO info;
	if (!GetConsoleCursorInfo(h, &info))
		return 0;
	lua_pushinteger(L, info.dwSize);
	lua_pushboolean(L, info.bVisible);
	return 2;
}

static int GetConsoleCursorSize(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	CONSOLE_CURSOR_INFO info;
	if (!GetConsoleCursorInfo(h, &info))
		return 0;
	lua_pushinteger(L, info.dwSize);
	return 1;
}

static int GetConsoleCursorVisible(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	CONSOLE_CURSOR_INFO info;
	if (!GetConsoleCursorInfo(h, &info))
		return 0;
	lua_pushboolean(L, info.bVisible);
	return 1;
}

static int GetCurrentConsoleFont(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	BOOL maximum = lua_toboolean(L, 2);
	CONSOLE_FONT_INFO info;
	if (!GetCurrentConsoleFont(h, maximum, &info))
		return 0;
	lua_createtable(L, 0, 2);
	lua_pushnumber(L, info.nFont);
	lua_setfield(L, -2, "Font");
	lua_createtable(L, 0, 2);
	lua_pushnumber(L, info.dwFontSize.X);
	lua_setfield(L, -2, "X");
	lua_pushnumber(L, info.dwFontSize.Y);
	lua_setfield(L, -2, "Y");
	lua_setfield(L, -2, "Size");
	return 1;
}

//static int GetCurrentConsoleFontEx(lua_State *L)
//static int SetCurrentConsoleFontEx(lua_State *L)

static int GetConsoleHistoryInfo(lua_State *L)
{
	CONSOLE_HISTORY_INFO info;
	info.cbSize = sizeof(CONSOLE_HISTORY_INFO);
	if (!GetConsoleHistoryInfo(&info))
		return 0;
	lua_createtable(L, 0, 3);
	lua_pushnumber(L, info.HistoryBufferSize);
	lua_setfield(L, -2, "HistoryBufferSize");
	lua_pushnumber(L, info.NumberOfHistoryBuffers);
	lua_setfield(L, -2, "NumberOfHistoryBuffers");
	lua_pushnumber(L, info.dwFlags);
	lua_setfield(L, -2, "Flags");
	return 1;
}

static int SetConsoleHistoryInfo(lua_State *L)
{
	CONSOLE_HISTORY_INFO info;
	info.cbSize = sizeof(CONSOLE_HISTORY_INFO);
	if (!GetConsoleHistoryInfo(&info))
		return 0;
	if (lua_istable(L, 1))
	{
		lua_getfield(L, 1, "HistoryBufferSize");
		info.HistoryBufferSize = UINT(luaL_optinteger(L, -1, info.HistoryBufferSize));
		lua_pop(L, 1);
		lua_getfield(L, 1, "NumberOfHistoryBuffers");
		info.NumberOfHistoryBuffers = UINT(luaL_optinteger(L, -1, info.NumberOfHistoryBuffers));
		lua_pop(L, 1);
		lua_getfield(L, 1, "Flags");
		info.dwFlags = DWORD(luaL_optinteger(L, -1, info.dwFlags));
		lua_pop(L, 1);
	}
	lua_pushboolean(L, SetConsoleHistoryInfo(&info));
	return 1;
}

//static int GetConsoleFontSize(lua_State *L)

static int GetConsoleSelectionInfo(lua_State *L)
{
	CONSOLE_SELECTION_INFO info;
	if (!GetConsoleSelectionInfo(&info))
		return 0;
	lua_createtable(L, 0, 3);
	lua_pushinteger(L, info.dwFlags);
	lua_setfield(L, -2, "Flags");
	lua_createtable(L, 0, 2);
	lua_pushinteger(L, info.dwSelectionAnchor.X);
	lua_setfield(L, -2, "X");
	lua_pushinteger(L, info.dwSelectionAnchor.Y);
	lua_setfield(L, -2, "Y");
	lua_setfield(L, -2, "SelectionAnchor");
	lua_createtable(L, 0, 4);
	lua_pushinteger(L, info.srSelection.Left);
	lua_setfield(L, -2, "Left");
	lua_pushinteger(L, info.srSelection.Top);
	lua_setfield(L, -2, "Top");
	lua_pushinteger(L, info.srSelection.Right);
	lua_setfield(L, -2, "Right");
	lua_pushinteger(L, info.srSelection.Bottom);
	lua_setfield(L, -2, "Bottom");
	lua_setfield(L, -2, "Selection");
	return 1;
}

static int SetConsoleActiveScreenBuffer(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	lua_pushboolean(L, SetConsoleActiveScreenBuffer(h));
	return 1;
}

//static int FlushConsoleInputBuffer(lua_State *L)

static int SetConsoleScreenBufferSize(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	COORD size = { 0, 0 };
	GetCoord(size, L, 2);
	lua_pushboolean(L, SetConsoleScreenBufferSize(h, size));
	return 1;
}

static int SetConsoleCursorPosition(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	COORD pos = { 0, 0 };
	GetCoord(pos, L, 2);
	lua_pushboolean(L, SetConsoleCursorPosition(h, pos));
	return 1;
}

static int SetConsoleCursorInfo(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	CONSOLE_CURSOR_INFO info;
	if (!GetConsoleCursorInfo(h, &info))
		return 0;
	if (lua_istable(L, 2))
	{
		lua_getfield(L, 2, "Size");
		if (lua_isnumber(L, -1))
			info.dwSize = lua_tointeger(L, -1);
		lua_pop(L, 1);
		lua_getfield(L, 2, "Visible");
		if (lua_isboolean(L, -1))
			info.bVisible = lua_toboolean(L, -1);
		lua_pop(L, 1);
	}
	else
	{
		if (lua_isnumber(L, 2))
			info.dwSize = luaL_checkinteger(L, 2);
		if (lua_isboolean(L, 3))
			info.bVisible = lua_toboolean(L, 3);
	}
	lua_pushboolean(L, SetConsoleCursorInfo(h, &info));
	return 1;
}

static int SetConsoleCursorSize(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	CONSOLE_CURSOR_INFO info;
	GetConsoleCursorInfo(h, &info);
	info.dwSize = luaL_checkinteger(L, 2);
	lua_pushboolean(L, SetConsoleCursorInfo(h, &info));
	return 1;
}

static int SetConsoleCursorVisible(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	CONSOLE_CURSOR_INFO info;
	GetConsoleCursorInfo(h, &info);
	info.bVisible = lua_toboolean(L, 3);
	lua_pushboolean(L, SetConsoleCursorInfo(h, &info));
	return 1;
}

static int ScrollConsoleScreenBufferA(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	int index = 2;
	SMALL_RECT scroll_rect = { 0, 0, 0, 0 };
	index += GetSmallRect(scroll_rect, L, index);
	SMALL_RECT clip_rect = { SHRT_MAX, SHRT_MAX, SHRT_MIN, SHRT_MIN };
	index += GetSmallRect(clip_rect, L, index);
	COORD dest_origin = { 0, 0 };
	index += GetCoord(dest_origin, L, index);
	CHAR_INFO fill = { 0, 0 };
	index += GetCharInfo(fill, L, index);
	lua_pushboolean(L, ScrollConsoleScreenBufferA(h, &scroll_rect,
		clip_rect.Left <= clip_rect.Left && clip_rect.Top <= clip_rect.Bottom ? &clip_rect : NULL,
		dest_origin, &fill));
	return 1;
}

static int SetConsoleWindowInfo(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	BOOL absolute = lua_toboolean(L, 2);
	SMALL_RECT window = { 0, 0, 0, 0 };
	GetSmallRect(window, L, 3);
	lua_pushboolean(L, SetConsoleWindowInfo(h, absolute, &window));
	return 1;
}

static int SetConsoleTextAttribute(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	WORD attrib = GetAttrib(L, 2);
	lua_pushboolean(L, SetConsoleTextAttribute(h, attrib));
	return 1;
}

//static int GenerateConsoleCtrlEvent(lua_State *L)

static int GetConsoleTitleA(lua_State *L)
{
	char title[256] = "";
	DWORD length = GetConsoleTitleA(title, sizeof(title));
	if (length == 0)
		return 0;
	lua_pushlstring(L, title, length);
	return 1;
}

static int GetConsoleOriginalTitleA(lua_State *L)
{
	char title[256] = "";
	DWORD length = GetConsoleOriginalTitleA(title, sizeof(title));
	if (length == 0)
		return 0;
	lua_pushlstring(L, title, length);
	return 1;
}

static int SetConsoleTitleA(lua_State *L)
{
	const char *title = luaL_checkstring(L, 1);
	lua_pushboolean(L, SetConsoleTitleA(title));
	return 1;
}

static int CreateConsoleScreenBuffer(lua_State *L)
{
	if (HANDLE h = CreateConsoleScreenBuffer(GENERIC_READ | GENERIC_WRITE, 0, NULL, CONSOLE_TEXTMODE_BUFFER, NULL))
	{
		PushHandle(L, h);
		return 1;
	}
	return 0;
}

///////////////
// consoleapi.h

static int GetConsoleMode(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	DWORD mode;
	if (!GetConsoleMode(h, &mode))
		return 0;
	lua_pushinteger(L, mode);
	return 1;
}

//static int GetNumberOfConsoleInputEvents(lua_State *L)
//static int PeekConsoleInputA(lua_State *L)

static int ReadConsoleA(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	char buffer[65536];
	DWORD length;
	if (!ReadConsoleA(h, &buffer, sizeof(buffer), &length, NULL))
		return 0;
	lua_pushlstring(L, buffer, length);
	return 1;
}

//static int ReadConsoleInputA(lua_State *L)
//static int SetConsoleCtrlHandler(lua_State *L)

static int SetConsoleMode(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	DWORD mode = DWORD(luaL_checkinteger(L, 2));
	lua_pushboolean(L, SetConsoleMode(h, mode));
	return 1;
}

static int WriteConsoleA(lua_State *L)
{
	HANDLE h = GetHandle(L, 1);
	size_t length = 0;
	const char * text = luaL_checklstring(L, 2, &length);
	DWORD written;
	WriteConsoleA(h, text, length, &written, NULL);
	lua_pushinteger(L, written);
	return 1;
}

////////
// other

static void DispatchInputEvents(lua_State *L, HANDLE hIn)
{
	// if there are any pending input events...
	DWORD numEvents = 0;
	while ((gameState == GAME_PAUSED) || (GetNumberOfConsoleInputEvents(hIn, &numEvents) && numEvents > 0))
	{
		// get the next input event
		INPUT_RECORD input;
		ReadConsoleInput(hIn, &input, 1, &numEvents);
		switch (input.EventType)
		{
		case KEY_EVENT:
			{
				const KEY_EVENT_RECORD &keyevent = input.Event.KeyEvent;
				lua_getglobal(L, keyevent.bKeyDown
							  ? "KeyPressed" : "KeyReleased");
				if (lua_isfunction(L, -1))
				{
					lua_pushinteger(L, keyevent.wRepeatCount);
					lua_pushinteger(L, keyevent.wVirtualKeyCode);
					lua_pushinteger(L, keyevent.wVirtualScanCode);
					lua_pushlstring(L, &keyevent.uChar.AsciiChar, 1);
					lua_pushinteger(L, keyevent.dwControlKeyState);
					LuaCheckStatus(lua_pcall(L, 5, 0, 0), L, keyevent.bKeyDown
								   ? "Lua script KeyPressed error:\n%s\n"
								   : "Lua script KeyReleased error:\n%s\n");
				}
				else
				{
					// default to escape quit
					if (keyevent.bKeyDown && keyevent.wVirtualKeyCode == VK_ESCAPE)
						gameState = GAME_QUIT;
					lua_pop(L, 1);
				}
			}
			break;

		case MOUSE_EVENT:
			{
				const MOUSE_EVENT_RECORD &mouseevent = input.Event.MouseEvent;
				if (mouseevent.dwEventFlags & MOUSE_MOVED)
				{
					lua_getglobal(L, "MouseMoved");
					if (lua_isfunction(L, -1))
					{
						lua_pushinteger(L, mouseevent.dwMousePosition.X);
						lua_pushinteger(L, mouseevent.dwMousePosition.Y);
						LuaCheckStatus(lua_pcall(L, 2, 0, 0), L, "Lua script MouseMoved errorL\n%s\n");
					}
					else
					{
						lua_pop(L, 1);
					}
				}
				if (LOWORD(mouseevent.dwButtonState) != mouseButtons)
				{
					for (int i = 0; i < 16; ++i)
					{
						bool isPressed = ((mouseevent.dwButtonState >> i) & 1) != 0;
						bool wasPressed = ((mouseButtons >> i) & 1) != 0;
						if (isPressed != wasPressed)
						{
							lua_getglobal(L, isPressed ? "MousePressed" : "MouseReleased");
							if (lua_isfunction(L, -1))
							{
								lua_pushinteger(L, i);
								LuaCheckStatus(lua_pcall(L, 1, 0, 0), L, isPressed
											   ? "Lua script MousePressed errorL\n%s\n"
											   : "Lua script MouseReleased errorL\n%s\n");
							}
							else
							{
								lua_pop(L, 1);
							}
						}
					}
					mouseButtons = LOWORD(mouseevent.dwButtonState);
				}
				if (mouseevent.dwEventFlags & MOUSE_WHEELED)
				{
					lua_getglobal(L, "MouseWheel");
					if (lua_isfunction(L, -1))
					{
						lua_pushinteger(L, HIWORD(mouseevent.dwButtonState));
						LuaCheckStatus(lua_pcall(L, 1, 0, 0), L, "Lua script MouseMoved error\n%s\n");
					}
					else
					{
						lua_pop(L, 1);
					}
				}
			}
			break;

		case WINDOW_BUFFER_SIZE_EVENT:
			{
				const WINDOW_BUFFER_SIZE_RECORD &sizeevent = input.Event.WindowBufferSizeEvent;
				lua_getglobal(L, "BufferSize");
				if (lua_isfunction(L, -1))
				{
					lua_pushinteger(L, sizeevent.dwSize.X);
					lua_pushinteger(L, sizeevent.dwSize.Y);
					LuaCheckStatus(lua_pcall(L, 2, 0, 0), L, "Lua script BufferSize error\n%s\n");
				}
				else
				{
					lua_pop(L, 1);
				}
			}
			break;

		case FOCUS_EVENT:
			{
				const FOCUS_EVENT_RECORD &focusevent = input.Event.FocusEvent;
				lua_getglobal(L, "SetFocus");
				if (lua_isfunction(L, -1))
				{
					lua_pushboolean(L, focusevent.bSetFocus);
					LuaCheckStatus(lua_pcall(L, 1, 0, 0), L, "Lua script SetFocus error\n%s\n");
				}
				else
				{
					lua_pop(L, 1);
					
					// default focus event handler
					gameState = focusevent.bSetFocus ? GAME_RUNNING : GAME_PAUSED;
				}
			}
			break;
		}
	}
}

// Lua script utils functions
static const luaL_Reg sLuaScriptUtils[] = {
	{ "Quit", Quit },
	{ "Pause", Pause },
	{ "Sleep", Sleep },
	{ "GetMillis", GetMillis },
	{ "GetLastError", GetLastError },
	{ "NewCharInfoBuffer", NewCharInfoBuffer },
	{ "CloseHandle", CloseHandle },
	///////////
	// wincon.h
	//{ "WriteInput", WriteConsoleInputA },
	{ "ReadOutput", ReadConsoleOutputA },
	{ "WriteOutput", WriteConsoleOutputA },
	{ "ReadOutputCharacter", ReadConsoleOutputCharacterA },
	{ "ReadOutputAttribute", ReadConsoleOutputAttribute },
	{ "WriteOutputCharacter", WriteConsoleOutputCharacterA },
	{ "WriteOutputAttribute", WriteConsoleOutputAttribute },
	{ "FillOutputCharacter", FillConsoleOutputCharacterA },
	{ "FillOutputAttribute", FillConsoleOutputAttribute },
	{ "GetScreenBufferInfo", GetConsoleScreenBufferInfo },
	{ "GetScreenBufferInfoEx", GetConsoleScreenBufferInfoEx },
	{ "GetScreenBufferSize", GetConsoleScreenBufferSize },
	{ "GetCursorPosition", GetConsoleCursorPosition },
	{ "GetTextAttribute", GetConsoleTextAttribute },
	{ "GetScreenBufferRect", GetConsoleScreenBufferRect },
	{ "GetPopupAttributes", GetConsolePopupAttributes },
	{ "GetFullscreenSupported", GetConsoleFullscreenSupported },
	{ "GetColor", GetConsoleColorTable },
	{ "SetScreenBufferInfoEx", SetConsoleScreenBufferInfoEx },
	{ "SetColor", SetConsoleColorTable },
	{ "GetLargestWindowSize", GetLargestConsoleWindowSize },
	{ "GetCursorInfo", GetConsoleCursorInfo },
	{ "GetCursorSize", GetConsoleCursorSize },
	{ "GetCursorVisible", GetConsoleCursorVisible },
	{ "GetCurrentFont", GetCurrentConsoleFont },
	//{ "GetCurrentFontEx", GetCurrentConsoleFontEx },
	//{ "SetCurrentFontEx", SetCurrentConsoleFontEx },
	{ "GetHistoryInfo", GetConsoleHistoryInfo },
	{ "SetHistoryInfo", SetConsoleHistoryInfo },
	//{ "GetFontSize", GetConsoleFontSize },
	{ "GetSelectionInfo", GetConsoleSelectionInfo },
	{ "SetActiveScreenBuffer", SetConsoleActiveScreenBuffer },
	//{ "FlushInputBuffer", FlushConsoleInputBuffer },
	{ "SetScreenBufferSize", SetConsoleScreenBufferSize },
	{ "SetCursorPosition", SetConsoleCursorPosition },
	{ "SetCursorInfo", SetConsoleCursorInfo },
	{ "SetCursorSize", SetConsoleCursorSize },
	{ "SetCursorVisible", SetConsoleCursorVisible },
	{ "ScrollScreenBuffer", ScrollConsoleScreenBufferA },
	{ "SetWindowInfo", SetConsoleWindowInfo },
	{ "SetTextAttribute", SetConsoleTextAttribute },
	//{ "GenerateCtrlEvent", GenerateConsoleCtrlEvent },
	{ "GetTitle", GetConsoleTitleA },
	{ "GetOriginalTitle", GetConsoleOriginalTitleA },
	{ "SetTitle", SetConsoleTitleA },
	{ "CreateScreenBuffer", CreateConsoleScreenBuffer },
	///////////////
	// consoleapi.h
	{ "GetMode", GetConsoleMode },
	//{ "GetNumberOfInputEvents", GetNumberOfConsoleInputEvents },
	//{ "PeekInput", PeekConsoleInputA },
	{ "Read", ReadConsoleA },
	//{ "ReadInput", ReadConsoleInputA },
	//{ "SetCtrlHandler", SetConsoleCtrlHandler },
	{ "SetMode", SetConsoleMode },
	{ "Write", WriteConsoleA },

	{ NULL, NULL }
};

int main(int argc, char* argv[])
{
	// Set up the handles for reading/writing:
	hOut = GetStdHandle(STD_OUTPUT_HANDLE);
	hIn = GetStdHandle(STD_INPUT_HANDLE);

	if (argc < 2)
	{
		ConsolePrint("No Lua script specified\n");
		return 1;
	}

	// Create a new Lua state with default memory allocator
	lua_State *L = luaL_newstate();
	if (!L)
		return 1;

	// Load standard libraries
	// (excluding io and os for security; see linit.c)
	luaL_openlibs(L);

	// Register a panic handler
	lua_atpanic(L, LuaPanic);

	// Replace the print function
	lua_register(L, "print", LuaPrint);

	// Add a warning function
	lua_register(L, "warning", LuaWarning);

	// Add a debug print function
	lua_register(L, "debugprint", LuaDebugPrint);

	// Register our functions
	// (into the global table)
	lua_pushglobaltable(L);
	luaL_setfuncs(L, sLuaScriptUtils, 0);

	// Create a metatable for handles
	lua_pushlightuserdata(L, NULL);
	luaL_newmetatable(L, "HANDLE");
	lua_pushstring(L, "HANDLE");
	lua_setfield(L, -2, "__type");
	lua_pushcfunction(L, HANDLE_ToString);
	lua_setfield(L, -2, "__tostring");
	lua_setmetatable(L, -2);

	// Create a metatable for CHAR_INFO_Buffer
	luaL_newmetatable(L, "CHAR_INFO_Buffer");
	lua_pushstring(L, "CHAR_INFO_Buffer");
	lua_setfield(L, -2, "__type");
	lua_pushvalue(L, -1);
	lua_setfield(L, - 2, "__index");
	lua_pushvalue(L, -1);
	lua_setfield(L, -2, "__metatable");
	lua_pushcfunction(L, CHAR_INFO_Buffer_Get);
	lua_setfield(L, -2, "Get");
	lua_pushcfunction(L, CHAR_INFO_Buffer_Set);
	lua_setfield(L, -2, "Set");

	// Default console handles
	PushHandle(L, hOut);
	lua_setglobal(L, "hOut");
	PushHandle(L, hIn);
	lua_setglobal(L, "hIn");

	// Initialize performance counter
	QueryPerformanceFrequency(&perfFrequency);
	QueryPerformanceCounter(&perfStart);

	// Return value for the executable
	int retval = 0;

	// Try to load and run the script
	if (LuaCheckStatus(luaL_loadfile(L, argv[1]), L, "Lua script load error:\n%s\n") )
	{
		// push additional arguments as script parameters
		for (int i = 2; i < argc; ++i)
			lua_pushstring(L, argv[i]);

		if (LuaCheckStatus(lua_pcall(L, argc - 2, 0, 0), L, "Lua script run error:\n%s\n"))
		{
			// Game loop
			while (gameState != GAME_QUIT)
			{
				// dispatch all input events
				DispatchInputEvents(L, hIn);

				// if the script has an Update function...
				lua_getglobal(L, "Update");
				if (lua_isfunction(L, -1))
				{
					LuaCheckStatus(lua_pcall(L, 0, 0, 0), L, "Lua script Update error:\n%s\n");
				}
				else
				{
					lua_pop(L, 1);
				}
			}
		}
		else
		{
			// report error
			retval = 1;
		}
	}
	else
	{
		// report error
		retval = 1;
	}

	// Close Lua state
	if (L)
		lua_close(L);

	return retval;
}

